/*
 * Ensuring smooth padding to limit edge effects from padding and peridiosing.
 * This file is part of ufo-serge filter set.
 * Copyright (C) 2025 Serge Cohen
 *
 * This library is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation, either
 * version 3 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Serge Cohen <serge.cohen@cnrs.fr>
 */
#include "config.h"

#include <ufo/ufo-cl.h>

#include "ufo-smooth-pad-task.h"

// Temporarily :
// #include <stdio.h>

struct _UfoSmoothPadTaskPrivate {
  cl_context context;
  cl_kernel kernel;
  gsize in_width;   // The width of the input image
  gsize out_width;  // The width of the output image (>= in_width)
  gsize out_offset; // The offset of the output image (<= out_width - in_width)
  gsize in_center;  // When not 0, used to compute the value of out_offset
};

static void ufo_task_interface_init (UfoTaskIface *iface);

G_DEFINE_TYPE_WITH_CODE (UfoSmoothPadTask, ufo_smooth_pad_task, UFO_TYPE_TASK_NODE,
                         G_IMPLEMENT_INTERFACE (UFO_TYPE_TASK,
                                                ufo_task_interface_init)
			 G_ADD_PRIVATE(UfoSmoothPadTask))

// Trying to use a newer version to avoid warning about deprecation and using G_ADD_PRIVATE
// #define UFO_SMOOTH_PAD_TASK_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE((obj), UFO_TYPE_SMOOTH_PAD_TASK, UfoSmoothPadTaskPrivate))
#define UFO_SMOOTH_PAD_TASK_GET_PRIVATE(obj) (ufo_smooth_pad_task_get_instance_private((UfoSmoothPadTask*)(obj)))

  enum {
  PROP_0,
  PROP_OUT_WIDTH,
  PROP_IN_CENTER, // The position of the center in the inpgut image (will compute offset to place it centrally for output image.
  PROP_OUT_OFFSET, // The integer value of the offset (computed if PROP_IN_CENTER is provided).
  N_PROPERTIES
};

static GParamSpec *properties[N_PROPERTIES] = { NULL, };

UfoNode *
ufo_smooth_pad_task_new (void)
{
  return UFO_NODE (g_object_new (UFO_TYPE_SMOOTH_PAD_TASK, NULL));
}

static void
ufo_smooth_pad_task_setup (UfoTask *task,
			   UfoResources *resources,
			   GError **error)
{
  UfoSmoothPadTaskPrivate *priv;

  priv = UFO_SMOOTH_PAD_TASK_GET_PRIVATE (task);

  // Not sure those (in_width in particular) are properly set by the time we run those checks
  if ( priv->out_width < priv->in_width ) {
    g_set_error (error, UFO_TASK_ERROR, UFO_TASK_ERROR_SETUP,
                 "Output width %ld is not larger than input width %ld",
                 priv->out_width, priv->in_width);
    return;
  }
  if ( priv->out_width < (priv->in_width + priv->out_offset) ) {
    g_set_error (error, UFO_TASK_ERROR, UFO_TASK_ERROR_SETUP,
                 "Output width %ld is not larger than input width plus offset %ld",
                 priv->out_width, (priv->in_width + priv->out_offset));
    return;
  }

  priv->kernel = ufo_resources_get_kernel (resources, "smooth-pad.cl", "smooth_pad", NULL, error);

  if (priv->kernel != NULL)
    UFO_RESOURCES_CHECK_AND_SET (clRetainKernel (priv->kernel), error);
}

static void
ufo_smooth_pad_task_get_requisition (UfoTask *task,
				     UfoBuffer **inputs,
				     UfoRequisition *requisition,
				     GError **error)
{
  UfoSmoothPadTaskPrivate *priv;

  priv = UFO_SMOOTH_PAD_TASK_GET_PRIVATE (task);

  // Using the requisition of the input, then editing the width (to mach out_width)
  ufo_buffer_get_requisition(inputs[0], requisition);
  priv->in_width = requisition->dims[0]; // So we know the input size
  requisition->dims[0] = priv->out_width;

  if (priv->in_center) { // in_center is not 0, use it to compute out_offset
    priv->out_offset = (priv->out_width >> 1) - (priv->in_width - priv->in_center);
  }
}

static guint
ufo_smooth_pad_task_get_num_inputs (UfoTask *task)
{
  return 1;
}

static guint
ufo_smooth_pad_task_get_num_dimensions (UfoTask *task,
					guint input)
{
  return 2;
}

static UfoTaskMode
ufo_smooth_pad_task_get_mode (UfoTask *task)
{
  return UFO_TASK_MODE_PROCESSOR | UFO_TASK_MODE_GPU;
}

static gboolean
ufo_smooth_pad_task_process (UfoTask *task,
			     UfoBuffer **inputs,
			     UfoBuffer *output,
			     UfoRequisition *requisition)
{
  UfoSmoothPadTaskPrivate *priv = UFO_SMOOTH_PAD_TASK_GET_PRIVATE (task);

  // Getting GPU resources to perform the computation :
  UfoGpuNode *node;
  UfoProfiler *profiler;
  cl_command_queue cmd_queue;
  cl_mem in_mem;
  cl_mem out_mem;

  UfoRequisition in_req;

  ufo_buffer_get_requisition(inputs[0], &in_req);
  node = UFO_GPU_NODE (ufo_task_node_get_proc_node (UFO_TASK_NODE (task)));
  cmd_queue = ufo_gpu_node_get_cmd_queue (node);

  // making sure the input image is in GPU memory :
  in_mem = ufo_buffer_get_device_array (inputs[0], cmd_queue);
  out_mem = ufo_buffer_get_device_array (output, cmd_queue);

  // Setting up the kernel to run :
  cl_uint in_width, out_width, out_offset;
  in_width = in_req.dims[0];
  out_width = requisition->dims[0];
  out_offset = (cl_uint)(priv->out_offset);
  UFO_RESOURCES_CHECK_CLERR (clSetKernelArg (priv->kernel, 0, sizeof (cl_mem), &in_mem));
  UFO_RESOURCES_CHECK_CLERR (clSetKernelArg (priv->kernel, 1, sizeof (cl_mem), &out_mem));
  UFO_RESOURCES_CHECK_CLERR (clSetKernelArg (priv->kernel, 2, sizeof (cl_uint), &in_width));
  UFO_RESOURCES_CHECK_CLERR (clSetKernelArg (priv->kernel, 3, sizeof (cl_uint), &out_width));
  UFO_RESOURCES_CHECK_CLERR (clSetKernelArg (priv->kernel, 4, sizeof (cl_uint), &out_offset));

  profiler = ufo_task_node_get_profiler (UFO_TASK_NODE (task));
  size_t global_size[3];
  global_size[0] = requisition->dims[1];
  global_size[1] = global_size[2] = 0;
  /*
  fprintf(stderr, "Calling smooth_pad kernel with arguments in_width=%d, out_width=%d, out_offset=%d\n"
	  , in_width, out_width, out_offset);
  */
  ufo_profiler_call (profiler, cmd_queue, priv->kernel, 1, global_size, NULL);

  return TRUE;
}


static void
ufo_smooth_pad_task_set_property (GObject *object,
				  guint property_id,
				  const GValue *value,
				  GParamSpec *pspec)
{
  UfoSmoothPadTaskPrivate *priv = UFO_SMOOTH_PAD_TASK_GET_PRIVATE (object);

  switch (property_id) {
  case PROP_OUT_WIDTH:
    priv->out_width = g_value_get_uint(value);
    break;
  case PROP_IN_CENTER:
     priv->in_center = g_value_get_uint(value);
    break;
  case PROP_OUT_OFFSET:
     priv->out_offset = g_value_get_uint(value);
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    break;
  }
}

static void
ufo_smooth_pad_task_get_property (GObject *object,
				  guint property_id,
				  GValue *value,
				  GParamSpec *pspec)
{
  UfoSmoothPadTaskPrivate *priv = UFO_SMOOTH_PAD_TASK_GET_PRIVATE (object);

  switch (property_id) {
  case PROP_OUT_WIDTH:
    g_value_set_uint(value, priv->out_width);
    break;
  case PROP_IN_CENTER:
    g_value_set_uint(value, priv->in_center);
    break;
  case PROP_OUT_OFFSET:
    g_value_set_uint(value, priv->out_offset);
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    break;
  }
}

static void
ufo_smooth_pad_task_finalize (GObject *object)
{
  UfoSmoothPadTaskPrivate *priv = UFO_SMOOTH_PAD_TASK_GET_PRIVATE (object);

  if ( priv->kernel ) {
    UFO_RESOURCES_CHECK_CLERR (clReleaseKernel (priv->kernel));
    priv->kernel = NULL;
  }

  if (priv->context) {
    UFO_RESOURCES_CHECK_CLERR (clReleaseContext (priv->context));
    priv->context = NULL;
  }

  G_OBJECT_CLASS (ufo_smooth_pad_task_parent_class)->finalize (object);
}

static void
ufo_task_interface_init (UfoTaskIface *iface)
{
  iface->setup = ufo_smooth_pad_task_setup;
  iface->get_num_inputs = ufo_smooth_pad_task_get_num_inputs;
  iface->get_num_dimensions = ufo_smooth_pad_task_get_num_dimensions;
  iface->get_mode = ufo_smooth_pad_task_get_mode;
  iface->get_requisition = ufo_smooth_pad_task_get_requisition;
  iface->process = ufo_smooth_pad_task_process;
}

static void
ufo_smooth_pad_task_class_init (UfoSmoothPadTaskClass *klass)
{
  GObjectClass *oclass = G_OBJECT_CLASS (klass);

  oclass->set_property = ufo_smooth_pad_task_set_property;
  oclass->get_property = ufo_smooth_pad_task_get_property;
  oclass->finalize = ufo_smooth_pad_task_finalize;

  properties[PROP_OUT_WIDTH] =
    g_param_spec_uint ("out-width",
		       "The width of the images after padding",
		       "Should be larger than the input image width, and sufficient to accomodate both input image and offset",
		       0,
		       G_MAXUINT,
		       0,
		       G_PARAM_READWRITE);

  properties[PROP_IN_CENTER] =
    g_param_spec_uint ("in-center",
		       "The (integer) value of the axis position in x",
		       "If set (and not 0), will compute the out-offset so that it puts the in-center at the center of the output, padded, frame",
		       0,
		       G_MAXUINT,
		       0,
		       G_PARAM_READWRITE);

  properties[PROP_OUT_OFFSET] =
    g_param_spec_uint ("out-offset",
		       "The number of pixel to offset the input image in the output image",
		       "Only used when in-center is set to 0 (otherwise the offset is derived from in-center), in which case the input image is offseted by out-offset pixels to the right to produce the output image. In any case, this should not lead to the input image being cropped on its right edge.",
		       0,
		       G_MAXUINT,
		       0,
		       G_PARAM_READWRITE);

  for (guint i = PROP_0 + 1; i < N_PROPERTIES; i++)
    g_object_class_install_property (oclass, i, properties[i]);

  // No more needed since using the G_ADD_PRIVATE macro in the G_DEFINE_TYPE_WITH_CODE call
  //  g_type_class_add_private (oclass, sizeof(UfoSmoothPadTaskPrivate));
}

static void
ufo_smooth_pad_task_init(UfoSmoothPadTask *self)
{
  self->priv = UFO_SMOOTH_PAD_TASK_GET_PRIVATE(self);
}
