/*
 * Copyright (c) 2017, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

/*
 * The certificates and corresponding private keys used by the test.
 * All of certificates uses relative weak key size and hash algorithm, then
 * all JDK releases can load them. Accordingly, a custom java.security file is
 * provided to make sure such weak key sizes and algorithms are not blocked by
 * any JDK build.
 */
public enum Cert {

    // This certificate is generated by the below command:
    // openssl req -x509 -newkey rsa:1024 -days 7300 \
    //     -subj "/CN=RSA_SHA1_1024" -sha1 \
    //     -keyout key.pem -out cert.pem
    RSA_SHA1_1024(
            SignatureAlgorithm.RSA,
            "-----BEGIN CERTIFICATE-----\n" +
            "MIIB/jCCAWegAwIBAgIJANPuKkD7/jxkMA0GCSqGSIb3DQEBBQUAMBgxFjAUBgNV\n" +
            "BAMMDVJTQV9TSEExXzEwMjQwHhcNMTcwOTA3MDIwNTM0WhcNMzcwOTAyMDIwNTM0\n" +
            "WjAYMRYwFAYDVQQDDA1SU0FfU0hBMV8xMDI0MIGfMA0GCSqGSIb3DQEBAQUAA4GN\n" +
            "ADCBiQKBgQC3v7UeIxD5bdv4mqwcpah7sNxpI3IxUFzI2ao1g1jVzDPZt9Zawa3K\n" +
            "H+m9al1Fg2X1dyNeRlbiXavcIZOQwZqNj08zJEwAdICP8iOnXQ2HUv5cpzArOPTu\n" +
            "GY3flhf39xgiWsSdfb+cP0QsWNagNU8EtebbHndv8W+2K5JEdlpwQQIDAQABo1Aw\n" +
            "TjAdBgNVHQ4EFgQU32KqdiGyzg39chNt/OwQzGOlUyAwHwYDVR0jBBgwFoAU32Kq\n" +
            "diGyzg39chNt/OwQzGOlUyAwDAYDVR0TBAUwAwEB/zANBgkqhkiG9w0BAQUFAAOB\n" +
            "gQAWx8y45IIWWhy44cuQs0qcSDQihIvhXB3pvlpCNdfsSrVoaaH8lrOVjTC718ip\n" +
            "fE1sF8I9niLHUg8WrAzdQRDsKyUhDUhEEJ7w1ffxwf8bcI9+NgWwEix0Dazzkub8\n" +
            "2IRXuZ3dGwzoI54XtxvKMFH86nJEj4M/XQGrc9bnlhcn4g==\n" +
            "-----END CERTIFICATE-----",
            "30820278020100300d06092a864886f70d0101010500048202623082025e0201" +
            "0002818100b7bfb51e2310f96ddbf89aac1ca5a87bb0dc69237231505cc8d9aa" +
            "358358d5cc33d9b7d65ac1adca1fe9bd6a5d458365f577235e4656e25dabdc21" +
            "9390c19a8d8f4f33244c0074808ff223a75d0d8752fe5ca7302b38f4ee198ddf" +
            "9617f7f718225ac49d7dbf9c3f442c58d6a0354f04b5e6db1e776ff16fb62b92" +
            "44765a7041020301000102818100b2c5afdf5c5a9d72c73b7eb0c9465b3fcc79" +
            "0549d946255bc0861555ef2eb503f1c67757f400cfa7019996123020fb906d5b" +
            "b66b789ffba90b16270cbd1fbfcf285a821dcdc78fd8f17f399eb231ce9724db" +
            "af60f9dd20f3e57bb4c0f9fdc9069589b82d442dd868d48c031eb782e27f9e70" +
            "8469f9b3d5b1b23cee5bf1b41781024100dec184ea77c2126c6bc0c01ba727b4" +
            "642587d63811240932334dc80c7976e0f715f156e52b352a25e5c52542af2b5f" +
            "68a29a9b68858f313c4375cc78ec03d859024100d32be8375f52cbe904002321" +
            "6977aee83fa88bf536d4052d2ed578727d7b7e5aeef91fc52b34c1b6638c00f0" +
            "4c6985fdaaa2d6e72adbcc7d10ed8bafff69da29024100ae8210acd6f13519b7" +
            "38a3c7862636ce1610daa3c5d9e3526e9acad3eafc54b57d7d3a44029b7dcf7e" +
            "b7f9beca1842806892929949b8aa2bb9f5b9202a55c0d1024100887dc0c2c9a2" +
            "429a823374818c2207b3a631d304d443867505e884c9bbc1ae9228146e2c8b18" +
            "b67ca52b411010d3c3ff89e366f454076dcd08bc01a5e8790ac102402321988a" +
            "2003e19c878791d402a7c0acdd1b6dd27203ed88f86a0e3a390ee57c0cd277f3" +
            "ea5df6440dbc8bdb4c8b3c28fc77e6991bc4ed3f4dc0619a5b953e8e"),

    // This certificate is generated by the below command:
    // openssl req -x509 -newkey rsa:1024 -days 7300 \
    //     -subj "/CN=www.example.com" -sha1 \
    //     -keyout key.pem -out cert.pem
    RSA_EXAMPLE_SHA1_1024(
            SignatureAlgorithm.RSA,
            "-----BEGIN CERTIFICATE-----\n" +
            "MIICAjCCAWugAwIBAgIJAK6TC9eDtZg4MA0GCSqGSIb3DQEBBQUAMBoxGDAWBgNV\n" +
            "BAMMD3d3dy5leGFtcGxlLmNvbTAeFw0xNzExMDIwNTA5NDRaFw0zNzEwMjgwNTA5\n" +
            "NDRaMBoxGDAWBgNVBAMMD3d3dy5leGFtcGxlLmNvbTCBnzANBgkqhkiG9w0BAQEF\n" +
            "AAOBjQAwgYkCgYEAtt5kxFTzJuoxJR2UgeXUxCw7TfL3FeK3lCrU3vruBe3XKKvF\n" +
            "oyCxf/B5ucm22gzMfOvJBWRg6KrNTrXGI1LtlmAYNDM5J0lK2N/neKOm3Qxe0d1W\n" +
            "AZ1lwgrMNirsWu+r4UPNMq5UohL5nqVU9WwVa12t0GF3er3k32tMTBqSclcCAwEA\n" +
            "AaNQME4wHQYDVR0OBBYEFNc8tKGfZdFyaY0ZslwGLt1kpRYAMB8GA1UdIwQYMBaA\n" +
            "FNc8tKGfZdFyaY0ZslwGLt1kpRYAMAwGA1UdEwQFMAMBAf8wDQYJKoZIhvcNAQEF\n" +
            "BQADgYEAc71ZO83YEw9WvhxDEng9tMYUhJnNZJss6+gfWjZ487aiEGnS+VgKsHWz\n" +
            "DBLBrYe9Ag5L9f1HtPNheUbnhhBbQ607jOG/wfmpi4VoU3myB5uxOfeAZdXDOB5x\n" +
            "bv3t7KcEhgmPjB/e123jrBK8qnAYmDlQVlkZScctB3I1OuA2Po4=\n" +
            "-----END CERTIFICATE-----",
            "30820277020100300d06092a864886f70d0101010500048202613082025d0201" +
            "0002818100b6de64c454f326ea31251d9481e5d4c42c3b4df2f715e2b7942ad4" +
            "defaee05edd728abc5a320b17ff079b9c9b6da0ccc7cebc9056460e8aacd4eb5" +
            "c62352ed96601834333927494ad8dfe778a3a6dd0c5ed1dd56019d65c20acc36" +
            "2aec5aefabe143cd32ae54a212f99ea554f56c156b5dadd061777abde4df6b4c" +
            "4c1a927257020301000102818048af52bc1acbdededd13d4930fa28b9441c47c" +
            "b222f5c6fc92df07676db3a815a61c9b51de0a03a347b10a609bd6459a0dd926" +
            "38877261686a5c6bb1ca9e8ea2373870af7685e7d6cebd66faba65af2ef04bd9" +
            "1244ae56900fcd6ce11207d8c4040176e4ba9fef3d563741a1027b229134cfe1" +
            "c0a90d9c8eba9ce6349835e769024100e82494b6f777c784ffc29298d033e11d" +
            "af46f0d464c4dbd950d46bcd697d0f0b49a77699f0111d408e8748f2b461ab8f" +
            "210071c9c20d8ecee3ae229cb9c3954b024100c9a976f0011fcdc0ca7fb2f679" +
            "974fa85d420c604ca7ff64fe4667a44f73088eef290d22195474039760e99325" +
            "3ca45ee444588b150467d14451d3c45dab0ba5024019df39d3ca70c703c39d63" +
            "c9342b1403c2ed1d1a0ec101df8e6a9e391e7099a4a068d187068261c8381a4b" +
            "bf00eb81bb49ea4ac439a4592e25a1daa9acea67510241008c4640007497bdd4" +
            "94473da26b33d06a29ecae9531dd4e2edf1cf42cfc42e53a1fac2b8183a3164c" +
            "053999600c6fe15a4c682a3b1cb482ceb33a4416fc9ce52d024100e4f08cd10a" +
            "5c8face0b20db86443d0a42e34dfdde236dae4f042a06dd3aff7ca159f8aa3b7" +
            "854df41d510148096155204f2bf46c4a96e271747a4126a66ade6c"),

    // This certificate is generated by the below commands:
    // openssl dsaparam -genkey 1024 -out key.pem
    // openssl req -x509 -new -key key.pem -days 7300 \
    //     -subj "/CN=DSA_SHA1_1024" -sha1 -out cert.pem
    DSA_SHA1_1024(
            SignatureAlgorithm.DSA,
            "-----BEGIN CERTIFICATE-----\n" +
            "MIICuzCCAnugAwIBAgIJAMAMLRrhQWQFMAkGByqGSM44BAMwGDEWMBQGA1UEAwwN\n" +
            "RFNBX1NIQTFfMTAyNDAeFw0xNzExMDIwNjA4MDRaFw0zNzEwMjgwNjA4MDRaMBgx\n" +
            "FjAUBgNVBAMMDURTQV9TSEExXzEwMjQwggG2MIIBKwYHKoZIzjgEATCCAR4CgYEA\n" +
            "8CspE1sE84pJ4YxzVHFEDNJvBaIxsbax03pDwNHr/ogP9PVwF9z1jT6hpC5WluHG\n" +
            "g5n5gqpF2XpBhX2fKm1qqZWRxNvHKo0+zzAhUqMrvRJqcjlL4ijXndHldt67/VKS\n" +
            "0eTKi9m64c+yJx80YYphCO5b93d2sTM29z8QZOlrbD8CFQCmttKnPAOk4uz8Z8cV\n" +
            "uPGeGOMB9wKBgCItgPpAjW0srIwCaDysDNpydX6hB+1NTy1gFYl24n8edLGbR0mZ\n" +
            "isteBd6LjMtgicRmtKZzKxW7igxoVvR3WHpTucFjms5NRNjPaj5wt3DxoXn4hyWk\n" +
            "LzMvDeBvi+jKJiO0jnQ3+1NDOlAQy6ukeH59/gxZ3UmcNxDlAQ/IYHcpA4GEAAKB\n" +
            "gEgvi72gL+zax7Y2hg4PL1PqZx2jFp0XlTIugiTrcsGytrAnn+/s2+3xVyVyvVMn\n" +
            "0z5yL5eP9cdGA7qV1+7n6KJ8jNAhLCBSiC6x5ekd88aTlqnmt5lstk4w0Q0zSa58\n" +
            "Hp6dCFg2Irk6Z9ERKaXJJBBS6reaFeATVROhN/LEEzzvo1AwTjAdBgNVHQ4EFgQU\n" +
            "jb+HHABclGNR4lpf19nHFZpfwPQwHwYDVR0jBBgwFoAUjb+HHABclGNR4lpf19nH\n" +
            "FZpfwPQwDAYDVR0TBAUwAwEB/zAJBgcqhkjOOAQDAy8AMCwCFDB3F/m6jsZdHaoy\n" +
            "1xTp2U8uHBO+AhQYzeJuJd8/qRSDVLs8mesE8TQg2g==\n" +
            "-----END CERTIFICATE-----",
            "3082014a0201003082012b06072a8648ce3804013082011e02818100f02b2913" +
            "5b04f38a49e18c735471440cd26f05a231b1b6b1d37a43c0d1ebfe880ff4f570" +
            "17dcf58d3ea1a42e5696e1c68399f982aa45d97a41857d9f2a6d6aa99591c4db" +
            "c72a8d3ecf302152a32bbd126a72394be228d79dd1e576debbfd5292d1e4ca8b" +
            "d9bae1cfb2271f34618a6108ee5bf77776b13336f73f1064e96b6c3f021500a6" +
            "b6d2a73c03a4e2ecfc67c715b8f19e18e301f7028180222d80fa408d6d2cac8c" +
            "02683cac0cda72757ea107ed4d4f2d60158976e27f1e74b19b4749998acb5e05" +
            "de8b8ccb6089c466b4a6732b15bb8a0c6856f477587a53b9c1639ace4d44d8cf" +
            "6a3e70b770f1a179f88725a42f332f0de06f8be8ca2623b48e7437fb53433a50" +
            "10cbaba4787e7dfe0c59dd499c3710e5010fc8607729041602146ef9db36045f" +
            "bcd8c7fd82ba29c5c5057ed11c7f"),

    // This certificate is generated by the below commands:
    // openssl dsaparam -genkey 1024 -out key.pem
    // openssl req -x509 -new -key key.pem -days 7300 \
    //     -subj "/CN=www.example.com" -sha1 -out cert.pem
    DSA_EXAMPLE_SHA1_1024(
            SignatureAlgorithm.DSA,
            "-----BEGIN CERTIFICATE-----\n" +
            "MIICwDCCAoCgAwIBAgIJAI5mKbdK5ZqyMAkGByqGSM44BAMwGjEYMBYGA1UEAwwP\n" +
            "d3d3LmV4YW1wbGUuY29tMB4XDTE3MTEwMjA1NDczOVoXDTM3MTAyODA1NDczOVow\n" +
            "GjEYMBYGA1UEAwwPd3d3LmV4YW1wbGUuY29tMIIBtzCCASwGByqGSM44BAEwggEf\n" +
            "AoGBANVGWRSlxVZQKlVrTDcU/6Mr8QFlR3kGKmkvdbTHH1EhcP7YlZ7CJ30VBDbN\n" +
            "LS2HvN3HHNooJ7hHBheL5Yz8EZIUa95TzPukZ1TmCo9fufR5i9HWj9Z8jLhyqx3l\n" +
            "iUZOYN9H0MSn4ftK6dr5oTz2ZGYDblXDCq6R8qZfuw1URFqrAhUArx0nmGEI/1S/\n" +
            "qyxnV4I6ItOntxMCgYEAxZKIZ/7aOGfzaQG2wRFdD/viHBZkkcxCsgmPUroQVUIw\n" +
            "dqmUnfYk8cb02LCevhhSwcjfocQsA3y1jufIUdWaHuIB9W3EsFJQNd/Byh9j/pRD\n" +
            "7zH/8lnBzJh2S7y10Vg840STVo5+ekZb4E+W7KK5gUaEQ6kAtUIIB0xjNz7RWs4D\n" +
            "gYQAAoGAPVQKWqJSlMrbU4XEsx50Ur8P84CwMnS7WcQNLnih1ScaK2BijgVj5Fny\n" +
            "9JZxITwj7XD7FWriq3kTjbydi3iAvrgVWij79x5Z7fTRCuoBVmtnAFkVGalwbGr2\n" +
            "ghz70y6hep2Evb1pRCrHjRkMaJFE5Y2CA7VbpKoat+j47/LkXJ2jUDBOMB0GA1Ud\n" +
            "DgQWBBSVjWy3SpaDfnFo+37mZJqX2aybzTAfBgNVHSMEGDAWgBSVjWy3SpaDfnFo\n" +
            "+37mZJqX2aybzTAMBgNVHRMEBTADAQH/MAkGByqGSM44BAMDLwAwLAIUd5NOlcfX\n" +
            "5rakT9H8UzlFcFQLr0MCFGrEYvlFUf/HJOH4FwXS2jEholBB\n" +
            "-----END CERTIFICATE-----",
            "3082014c0201003082012c06072a8648ce3804013082011f02818100d5465914" +
            "a5c556502a556b4c3714ffa32bf101654779062a692f75b4c71f512170fed895" +
            "9ec2277d150436cd2d2d87bcddc71cda2827b84706178be58cfc1192146bde53" +
            "ccfba46754e60a8f5fb9f4798bd1d68fd67c8cb872ab1de589464e60df47d0c4" +
            "a7e1fb4ae9daf9a13cf66466036e55c30aae91f2a65fbb0d54445aab021500af" +
            "1d27986108ff54bfab2c6757823a22d3a7b71302818100c5928867feda3867f3" +
            "6901b6c1115d0ffbe21c166491cc42b2098f52ba1055423076a9949df624f1c6" +
            "f4d8b09ebe1852c1c8dfa1c42c037cb58ee7c851d59a1ee201f56dc4b0525035" +
            "dfc1ca1f63fe9443ef31fff259c1cc98764bbcb5d1583ce34493568e7e7a465b" +
            "e04f96eca2b981468443a900b54208074c63373ed15ace0417021500abf47692" +
            "88c6ac41e2802e7eb7addba367339318"),

    // This certificate is generated by the below commands:
    // openssl ecparam -name prime256v1 -genkey -out key.pem
    // openssl req -new -key key.pem -x509 -nodes -days 7300 \
    //     -subj "/CN=ECDSA_SHA1_prime256v1" -sha1 -out cert.pem
    ECDSA_SHA1_PRIME256V1(
            SignatureAlgorithm.ECDSA,
            "-----BEGIN CERTIFICATE-----\n" +
            "MIIBhDCCASygAwIBAgIJAKW4wuujp9JbMAkGByqGSM49BAEwIDEeMBwGA1UEAwwV\n" +
            "RUNEU0FfU0hBMV9wcmltZTI1NnYxMB4XDTE3MDkwNzAyMTA0MVoXDTM3MDkwMjAy\n" +
            "MTA0MVowIDEeMBwGA1UEAwwVRUNEU0FfU0hBMV9wcmltZTI1NnYxMFkwEwYHKoZI\n" +
            "zj0CAQYIKoZIzj0DAQcDQgAEdbE+AMwsFBf73YXRVwsvsx2dMt1xgDxj/4pN+BfY\n" +
            "LWnO94beeZcrCJ1/N8CHmDOce7KRDR6/9kpi20wFAVXZ3KNQME4wHQYDVR0OBBYE\n" +
            "FA/hB2ODDNdz1JF08u2uhknhlsVoMB8GA1UdIwQYMBaAFA/hB2ODDNdz1JF08u2u\n" +
            "hknhlsVoMAwGA1UdEwQFMAMBAf8wCQYHKoZIzj0EAQNHADBEAiBNxv2L2FW+6+w/\n" +
            "QtDe+YSUNRj3F8QrpLkfGk7rVaOiHQIgVF2pWJ5ytg0pbCuO8Bh+UZ7zfZUD03s8\n" +
            "ZuIYW7RtMe0=\n" +
            "-----END CERTIFICATE-----",
            "308187020100301306072a8648ce3d020106082a8648ce3d030107046d306b02" +
            "010104204d901d5efd0e3def78d5307788a4c760115effce4b9e2c31ae5860b6" +
            "c11915aca1440342000475b13e00cc2c1417fbdd85d1570b2fb31d9d32dd7180" +
            "3c63ff8a4df817d82d69cef786de79972b089d7f37c08798339c7bb2910d1ebf" +
            "f64a62db4c050155d9dc"),

    // This certificate is generated by the below commands:
    // openssl ecparam -name prime256v1 -genkey -out key.pem
    // openssl req -new -key key.pem -x509 -nodes -days 7300 \
    //     -subj "/CN=www.example.com" -sha1 -out cert.pem
    ECDSA_EXAMPLE_SHA1_PRIME256V1(
            SignatureAlgorithm.ECDSA,
            "-----BEGIN CERTIFICATE-----\n" +
            "MIIBeDCCASCgAwIBAgIJAMxOXBpiJ5mDMAkGByqGSM49BAEwGjEYMBYGA1UEAwwP\n" +
            "d3d3LmV4YW1wbGUuY29tMB4XDTE3MTEwMjA1MTg0MVoXDTM3MTAyODA1MTg0MVow\n" +
            "GjEYMBYGA1UEAwwPd3d3LmV4YW1wbGUuY29tMFkwEwYHKoZIzj0CAQYIKoZIzj0D\n" +
            "AQcDQgAER9IyuwyrJ7X9DmIqGC3YNTlWBt4Fo/Y3RnlcxhTVxb/ZAYVNhqe4MbSM\n" +
            "2nsVnYMjjXXDav1plNKvmgGDf9s/saNQME4wHQYDVR0OBBYEFHNUTaIIEA89uNKH\n" +
            "OOUgJ981Qj5HMB8GA1UdIwQYMBaAFHNUTaIIEA89uNKHOOUgJ981Qj5HMAwGA1Ud\n" +
            "EwQFMAMBAf8wCQYHKoZIzj0EAQNHADBEAiBCW59S1nE15j8euO6/q9bM6J9Ci5xJ\n" +
            "WWAVznGGxnS/HgIgFaFKC31uxTXoBN7QN0yW/umQgJ0nsjwj7Pnxc0wNyw8=\n" +
            "-----END CERTIFICATE-----",
            "308187020100301306072a8648ce3d020106082a8648ce3d030107046d306b02" +
            "010104209aa3784cd0c1fe0553e59b3c7b8f08c8fdaffd94f34e2c1683243a79" +
            "7b64b673a1440342000447d232bb0cab27b5fd0e622a182dd835395606de05a3" +
            "f63746795cc614d5c5bfd901854d86a7b831b48cda7b159d83238d75c36afd69" +
            "94d2af9a01837fdb3fb1");

    public final SignatureAlgorithm signatureAlgorithm;
    public final String certMaterials;
    public final String privKeyMaterials;

    private Cert(
            SignatureAlgorithm signatureAlgorithm,
            String certMaterials,
            String privKeyMaterials) {
        this.signatureAlgorithm = signatureAlgorithm;
        this.certMaterials = certMaterials;
        this.privKeyMaterials = privKeyMaterials;
    }

    // Two certificates (mainCert and exampleCert) are selected to respect the
    // specified cipher suite. SNI-associated cases specify exampleCert as desired.
    public static Cert[] getCerts(String cipherSuite) {
        Cert mainCert = Cert.DSA_SHA1_1024;
        Cert exampleCert = Cert.DSA_EXAMPLE_SHA1_1024;
        if (cipherSuite.contains("_ECDHE_RSA_")) {
            mainCert = Cert.RSA_SHA1_1024;
            exampleCert = Cert.RSA_EXAMPLE_SHA1_1024;
        } else if (cipherSuite.contains("_EC")) {
            mainCert = Cert.ECDSA_SHA1_PRIME256V1;
            exampleCert = Cert.ECDSA_EXAMPLE_SHA1_PRIME256V1;
        } else if (cipherSuite.contains("_RSA")) {
            mainCert = Cert.RSA_SHA1_1024;
            exampleCert = Cert.RSA_EXAMPLE_SHA1_1024;
        }
        System.out.printf("mainCert=%s, exampleCert=%s%n",
                mainCert, exampleCert);
        return new Cert[] { mainCert, exampleCert };
    }
}

enum SignatureAlgorithm {

    RSA, DSA, ECDSA;
}
