//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Tests/Suite/Common/TestSuite.h
//! @brief     Provides TEST_F stances that define the BornAgain standard tests.
//!
//!            Each of these stances runs one standard test.
//!            The function run, defined in file RunTest.cpp, is the same for Core/Py/GUI std tests.
//!            It calls a function checkSimulation that is different for those three test suites.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_TESTS_SUITE_COMMON_TESTSUITE_H
#define BORNAGAIN_TESTS_SUITE_COMMON_TESTSUITE_H

#include "Base/Axis/Scale.h"
#include "Sample/ComponentBuilder/FormFactorComponents.h"
#include "Sample/ComponentBuilder/Profile2DComponents.h"
#include "Sample/Multilayer/MultiLayer.h"
#include "Sample/StandardSamples/ExemplarySamples.h"
#include "Sim/Background/ConstantBackground.h"
#include "Sim/Simulation/includeSimulations.h"
#include "Tests/GTestWrapper/google_test.h"
#include "Tests/SimFactory/MakeSimulations.h"

// The macro eps_direct_vs_python allows us to set different tolerances for
// - direct simulations (fully in C++),
// - simulations that are exported from C++ to Python, then executed as Python scripts.
// Python tests require larger tolerances because parameters are not written with
// full double precision to exported scripts.
#ifndef PYTHON_STD_TEST
#define eps_direct_vs_python(eps_direct, eps_python) eps_direct
#else
#define eps_direct_vs_python(eps_direct, eps_python) eps_python
#endif

TEST(TESTNAME, FormFactors)
{
    const double eps = eps_direct_vs_python(2e-13, 8e-11);
    for (const std::string& ffname : FormFactorComponents().keys()) {
        const IFormFactor* ff = FormFactorComponents().getItem(ffname)->clone();
        ASSERT(ff);
        std::unique_ptr<const MultiLayer> sample(
            ExemplarySamples::createParticleInVacuumWithFF(ff));
        ASSERT(sample);
        auto sim = test::makeSimulation::MiniGISAS(*sample);
        EXPECT_TRUE(runTest("FormFactors_" + ff->className(), *sim, eps));
    }
}

TEST(TESTNAME, FormFactorsWithAbsorption)
{
    const double eps = eps_direct_vs_python(8e-13, 3e-10);
    for (const std::string& ffname : FormFactorComponents().keys()) {
        const IFormFactor* ff = FormFactorComponents().getItem(ffname)->clone();
        ASSERT(ff);
        std::unique_ptr<const MultiLayer> sample(
            ExemplarySamples::createLayersWithAbsorptionWithFF(ff));
        ASSERT(sample);
        auto sim = test::makeSimulation::MiniGISAS_v2(*sample);
        EXPECT_TRUE(runTest("FormFactorsWithAbsorption_" + ff->className(), *sim, eps));
    }
}

TEST(TESTNAME, GISASAbsorptiveSLDLayers)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createLayersWithAbsorptionBySLD());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("GISASAbsorptiveSLDLayers", *sim, eps));
}

TEST(TESTNAME, CylindersAndPrisms)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createCylindersAndPrisms());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("CylindersAndPrisms", *sim, eps));
}

TEST(TESTNAME, RadialParacrystal)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createRadialParacrystal());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("RadialParacrystal", *sim, eps));
}

TEST(TESTNAME, HardDisk)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createHardDisk());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("HardDisk", *sim, eps));
}

TEST(TESTNAME, Basic2DParacrystal)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    for (const std::string& pdfname : Profile2DComponents().keys()) {
        const IProfile2D* pdf2 = Profile2DComponents().getItem(pdfname)->clone();
        ASSERT(pdf2);
        std::cout << pdf2->pythonConstructor() << std::endl;
        std::unique_ptr<const MultiLayer> sample(
            ExemplarySamples::createBasic2DParacrystalWithFTDis(pdf2));
        ASSERT(sample);
        auto sim = test::makeSimulation::MiniGISAS(*sample);
        EXPECT_TRUE(runTest("Basic2DParacrystal_" + pdf2->className(), *sim, eps));
    }
}

TEST(TESTNAME, HexParacrystal)
{
    const double eps = eps_direct_vs_python(8e-11, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createHexParacrystal());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("HexParacrystal", *sim, eps));
}

TEST(TESTNAME, Lattice1D)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createLattice1D());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("Lattice1D", *sim, eps));
}

TEST(TESTNAME, RectParacrystal)
{
    // TODO: investigate numeric integration, which has become problematic under Windows
    //       after moving code from Core/Tools to Base/Utils
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createRectParacrystal());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("RectParacrystal", *sim, 1e-9));
}

TEST(TESTNAME, CoreShellParticle)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createCoreShellParticle());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("CoreShellParticle", *sim, eps));
}

TEST(TESTNAME, CoreShellBoxRotateZandY)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createCoreShellBoxRotateZandY());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("CoreShellBoxRotateZandY", *sim, eps));
}

TEST(TESTNAME, MultiLayerWithRoughness)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createMultiLayerWithRoughness());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("MultiLayerWithRoughness", *sim, eps));
}

TEST(TESTNAME, SquareLattice2D)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createSquareLattice2D());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("SquareLattice2D", *sim, eps));
}

TEST(TESTNAME, CenteredSquareLattice2D)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createCenteredSquareLattice2D());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("CenteredSquareLattice2D", *sim, eps));
}

TEST(TESTNAME, RotatedSquareLattice2D)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createRotatedSquareLattice2D());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("RotatedSquareLattice2D", *sim, eps));
}

TEST(TESTNAME, FiniteSquareLattice2D)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createFiniteSquareLattice2D());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("FiniteSquareLattice2D", *sim, eps));
}

TEST(TESTNAME, RotatedPyramids)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createRotatedPyramids());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("RotatedPyramids", *sim, eps));
}

TEST(TESTNAME, ThickAbsorptiveSampleWithRoughness)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createThickAbsorptiveSample());
    auto sim = test::makeSimulation::ExtraLongWavelengthGISAS(*sample);
    EXPECT_TRUE(runTest("ThickAbsorptiveSampleWithRoughness", *sim, eps));
}

TEST(TESTNAME, Compound)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createCompound());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("Compound", *sim, eps));
}

TEST(TESTNAME, BoxCompositionRotateX)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createBoxCompositionRotateX());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("BoxCompositionRotateX", *sim, eps));
}

TEST(TESTNAME, BoxCompositionRotateY)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createBoxCompositionRotateY());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("BoxCompositionRotateY", *sim, eps));
}

TEST(TESTNAME, BoxCompositionRotateZ)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createBoxCompositionRotateZ());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("BoxCompositionRotateZ", *sim, eps));
}

TEST(TESTNAME, BoxCompositionRotateZandY)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createBoxCompositionRotateZandY());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("BoxCompositionRotateZandY", *sim, eps));
}

TEST(TESTNAME, BoxStackComposition)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createBoxStackComposition());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("BoxStackComposition", *sim, eps));
}

TEST(TESTNAME, MultipleLayout)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createMultipleLayout());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("MultipleLayout", *sim, eps));
}

TEST(TESTNAME, ApproximationDA)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createSizeDistributionDAModel());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("ApproximationDA", *sim, eps));
}

TEST(TESTNAME, ApproximationLMA)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createSizeDistributionLMAModel());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("ApproximationLMA", *sim, eps));
}

TEST(TESTNAME, ApproximationSSCA)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createSizeDistributionSSCAModel());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("ApproximationSSCA", *sim, eps));
}

TEST(TESTNAME, CosineRipple)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createCosineRipple());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("CosineRipple", *sim, eps));
}

TEST(TESTNAME, TriangularRipple)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createTriangularRipple());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("TriangularRipple", *sim, eps));
}

TEST(TESTNAME, AsymRipple)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createAsymRipple());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("AsymRipple", *sim, eps));
}

TEST(TESTNAME, Mesocrystal)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createMesocrystal());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("Mesocrystal", *sim, eps));
}

TEST(TESTNAME, CustomMorphology)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createCustomMorphology());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("CustomMorphology", *sim, eps));
}

TEST(TESTNAME, TransformBox)
{
    const double eps = eps_direct_vs_python(5e-13, 1e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createTransformBox());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("TransformBox", *sim, eps));
}

TEST(TESTNAME, MagneticParticleZeroField)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createMagneticParticleZeroField());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("MagneticParticleZeroField", *sim, eps));
}

TEST(TESTNAME, SlicedComposition)
{
    const double eps = eps_direct_vs_python(5e-13, 1e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createSlicedComposition());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("SlicedComposition", *sim, eps));
}

TEST(TESTNAME, MagneticSubstrateZeroField)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createMagneticSubstrateZeroField());
    auto sim = test::makeSimulation::MiniZPolarizedGISAS(*sample, "PP");
    EXPECT_TRUE(runTest("MagneticSubstrateZeroField", *sim, eps));
}

TEST(TESTNAME, MagneticRotation)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createMagneticRotation());
    auto sim = test::makeSimulation::MiniZPolarizedGISAS(*sample, "PM");
    EXPECT_TRUE(runTest("MagneticRotationZPM", *sim, eps));
}

TEST(TESTNAME, MagneticRotationUnpol)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createMagneticRotation());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("MagneticRotationUnpol", *sim, eps));
}

TEST(TESTNAME, MagneticSpheres)
{
    const double eps = eps_direct_vs_python(4e-3, 2e-10); // TODO mac-arm restore 5e-13
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createMagneticSpheres());
    auto sim = test::makeSimulation::MiniZPolarizedGISAS(*sample, "PM");
    EXPECT_TRUE(runTest("MagneticSpheres", *sim, eps));
}

TEST(TESTNAME, MagneticCylinders)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    for (const std::string polCase : {"PP", "MP", "PM", "MM"}) {
        std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createMagneticCylinders());
        auto sim = test::makeSimulation::MiniZPolarizedGISAS(*sample, polCase);
        EXPECT_TRUE(runTest("MagneticCylinders" + polCase, *sim, eps));
    }
}

TEST(TESTNAME, MagneticSpheresInMagLayer)
{
    const double eps = eps_direct_vs_python(4e-3, 2e-10); // TODO mac-arm restore 5e-13
    for (const std::string polCase : {"PP", "MP"}) {
        std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createMagneticLayer());
        auto sim = test::makeSimulation::MiniZPolarizedGISAS(*sample, polCase);
        EXPECT_TRUE(runTest("MagneticSpheresInMagLayer" + polCase, *sim, eps));
    }
}

TEST(TESTNAME, BeamDivergence)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createCylindersInBA());
    auto sim = test::makeSimulation::MiniGISASBeamDivergence(*sample);
    EXPECT_TRUE(runTest("BeamDivergence", *sim, eps));
}

TEST(TESTNAME, DetectorResolution)
{
    const double eps = eps_direct_vs_python(5e-13, 2e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createCylindersInBA());
    auto sim = test::makeSimulation::MiniGISASDetectorResolution(*sample);
    EXPECT_TRUE(runTest("DetectorResolution", *sim, eps));
}

TEST(TESTNAME, SimulationWithMasks)
{
    const double eps = eps_direct_vs_python(5e-13, 1e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createCylindersAndPrisms());
    auto sim = test::makeSimulation::GISASWithMasks(*sample);
    EXPECT_TRUE(runTest("SimulationWithMasks", *sim, eps));
}

TEST(TESTNAME, RectDetectorGeneric)
{
    const double eps = eps_direct_vs_python(5e-13, 1e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createCylindersInBA());
    auto sim = test::makeSimulation::RectDetectorGeneric(*sample);
    EXPECT_TRUE(runTest("RectDetectorGeneric", *sim, eps));
}

TEST(TESTNAME, RectDetectorPerpToSample)
{
    const double eps = eps_direct_vs_python(5e-13, 1e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createCylindersInBA());
    auto sim = test::makeSimulation::RectDetectorPerpToSample(*sample);
    EXPECT_TRUE(runTest("RectDetectorPerpToSample", *sim, eps));
}

TEST(TESTNAME, RectDetectorPerpToDirectBeam)
{
    const double eps = eps_direct_vs_python(5e-13, 1e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createCylindersInBA());
    auto sim = test::makeSimulation::RectDetectorPerpToDirectBeam(*sample);
    EXPECT_TRUE(runTest("RectDetectorPerpToDirectBeam", *sim, eps));
}

TEST(TESTNAME, RectDetectorPerpToReflectedBeam)
{
    const double eps = eps_direct_vs_python(5e-13, 1e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createCylindersInBA());
    auto sim = test::makeSimulation::RectDetectorPerpToReflectedBeam(*sample);
    EXPECT_TRUE(runTest("RectDetectorPerpToReflectedBeam", *sim, eps));
}

TEST(TESTNAME, LargeCylindersMonteCarlo)
{
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createLargeCylindersInDWBA());
    auto sim = test::makeSimulation::MiniGISASMonteCarlo(*sample);
    EXPECT_TRUE(runTest("LargeCylindersMonteCarlo", *sim, 5e-1));
}

TEST(TESTNAME, RectDetWithRoi)
{
    const double eps = eps_direct_vs_python(5e-13, 1e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createCylindersAndPrisms());
    auto sim = test::makeSimulation::RectDetWithRoi(*sample);
    EXPECT_TRUE(runTest("RectDetWithRoi", *sim, eps));
}

TEST(TESTNAME, BoxesWithSpecular)
{
    const double eps = eps_direct_vs_python(5e-13, 1e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createBoxesSquareLattice2D());
    auto sim = test::makeSimulation::MiniGISASSpecularPeak(*sample);
    EXPECT_TRUE(runTest("BoxesWithSpecular", *sim, eps));
}

TEST(TESTNAME, ConstantBackground)
{
    const double eps = eps_direct_vs_python(5e-13, 1e-10);
    auto* sample = ExemplarySamples::createCylindersInBA();
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    sim->setBackground(ConstantBackground(1e3));
    EXPECT_TRUE(runTest("ConstantBackground", *sim, eps));
}

TEST(TESTNAME, HomogeneousTiNiSample)
{
    const double eps = eps_direct_vs_python(5e-13, 1e-10);
    auto* sample = ExemplarySamples::createHomogeneousMultilayer();
    auto sim = test::makeSimulation::BasicSpecular(*sample, false);
    EXPECT_TRUE(runTest("HomogeneousTiNiSample", *sim, eps));
}

TEST(TESTNAME, HomogeneousTiNiSampleWithAbsorption)
{
    const double eps = eps_direct_vs_python(5e-13, 1e-10);
    auto* sample = ExemplarySamples::createPlainMultiLayerBySLD();
    auto sim = test::makeSimulation::BasicSpecular(*sample, false);
    EXPECT_TRUE(runTest("HomogeneousTiNiSampleWithAbsorption", *sim, eps));
}

TEST(TESTNAME, RoughnessInSpecular)
{
    auto* sample = ExemplarySamples::createMultiLayerWithRoughness();
    auto sim = test::makeSimulation::BasicSpecular(*sample, false);
    EXPECT_TRUE(runTest("RoughnessInSpecular", *sim, 2e-9));
}

TEST(TESTNAME, GaussianBeamFootprint)
{
    const double eps = eps_direct_vs_python(5e-13, 1e-10);
    auto* sample = ExemplarySamples::createHomogeneousMultilayer();
    auto sim = test::makeSimulation::SpecularWithGaussianBeam(*sample);
    EXPECT_TRUE(runTest("GaussianBeamFootprint", *sim, eps));
}

TEST(TESTNAME, SquareBeamFootprint)
{
    const double eps = eps_direct_vs_python(5e-13, 1e-10);
    auto* sample = ExemplarySamples::createHomogeneousMultilayer();
    auto sim = test::makeSimulation::SpecularWithSquareBeam(*sample);
    EXPECT_TRUE(runTest("SquareBeamFootprint", *sim, eps));
}

TEST(TESTNAME, SpecularDivergentBeam)
{
    const double eps = eps_direct_vs_python(5e-13, 1e-10);
    auto* sample = ExemplarySamples::createHomogeneousMultilayer();
    auto sim = test::makeSimulation::SpecularDivergentBeam(*sample);
    EXPECT_TRUE(runTest("SpecularDivergentBeam", *sim, eps));
}

//  ************************************************************************************************
//  TODO: broken under GUI
//  ************************************************************************************************

#ifndef GUI_STD_TEST

TEST(TESTNAME, RelativeResolutionTOF)
{
    const double eps = eps_direct_vs_python(5e-13, 1e-10);
    auto* sample = ExemplarySamples::createPlainMultiLayerBySLD();
    auto sim = test::makeSimulation::TOFRWithRelativeResolution(*sample);
    EXPECT_TRUE(runTest("RelativeResolutionTOF", *sim, eps));
}

TEST(TESTNAME, SphericalDetWithRoi)
{
    const double eps = eps_direct_vs_python(5e-13, 1e-10);
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createCylindersAndPrisms());
    auto sim = test::makeSimulation::SphericalDetWithRoi(*sample);
    EXPECT_TRUE(runTest("SphericalDetWithRoi", *sim, eps));
}

#endif // GUI_STD_TEST

//  ************************************************************************************************
//  TODO: broken under Python
//  ************************************************************************************************

#ifndef PYTHON_STD_TEST

TEST(TESTNAME, OffspecResonator)
{
    const double eps = eps_direct_vs_python(5e-13, 1e-10);
    auto* sample = ExemplarySamples::createResonator();
    auto sim = test::makeSimulation::MiniOffspec(*sample);
    EXPECT_TRUE(runTest("OffspecResonator", *sim, eps));
}

#endif // PYTHON_STD_TEST

//  ************************************************************************************************
//  TODO: broken under GUI and Python
//  ************************************************************************************************

#ifndef PYTHON_STD_TEST
#ifndef GUI_STD_TEST

TEST(TESTNAME, NCRoughnessInSpecular)
{
    auto* sample = ExemplarySamples::createMultiLayerWithNCRoughness();
    auto sim = test::makeSimulation::BasicSpecular(*sample, false);
    EXPECT_TRUE(runTest("NCRoughnessInSpecular", *sim, 2e-9));
}

TEST(TESTNAME, SuperLattice)
{
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createSuperLattice());
    auto sim = test::makeSimulation::MiniGISAS(*sample);
    EXPECT_TRUE(runTest("SuperLattice", *sim, 2e-10));
}

TEST(TESTNAME, SpecularWithSlicing1)
{
    // data almost fully agree with SpecularWithSlicing2, as verified by ConsistenceTest
    auto* sample = ExemplarySamples::createSlicedCylinders();
    auto sim = test::makeSimulation::BasicSpecular(*sample, false);
    EXPECT_TRUE(runTest("SpecularWithSlicing1", *sim, 1e-10));
}

TEST(TESTNAME, SpecularWithSlicing2)
{
    auto* sample = ExemplarySamples::createSLDSlicedCylinders();
    auto sim = test::makeSimulation::BasicSpecular(*sample, false);
    EXPECT_TRUE(runTest("SpecularWithSlicing2", *sim, 1e-10));
    auto simQ = test::makeSimulation::BasicSpecular(*sample, true);
    EXPECT_TRUE(runTest("SpecularWithSlicing2Q", *simQ, 1e-10));
}

TEST(TESTNAME, SpecularWithSlicing3)
{
    // data fully agree with SpecularWithSlicing2, as verified by ConsistenceTest
    auto* sample = ExemplarySamples::createAveragedSlicedCylinders();
    auto sim = test::makeSimulation::BasicSpecular(*sample, false);
    EXPECT_TRUE(runTest("SpecularWithSlicing3", *sim, 1e-10));
}

TEST(TESTNAME, InstrumentDefinitionComparison)
// also tests persistent result for the intensity fudge parameter
{
    auto* sample = ExemplarySamples::createPlainMultiLayerBySLD();
    auto sim = test::makeSimulation::BasicSpecular(*sample, false, 1.05);
    EXPECT_TRUE(runTest("InstrumentDefinitionComparison_0", *sim, 1e-10));
    auto simQ = test::makeSimulation::BasicSpecular(*sample, true, 1.05);
    EXPECT_TRUE(runTest("InstrumentDefinitionComparison_Q", *simQ, 1e-10));
}

TEST(TESTNAME, TOFResolutionComparison)
{
    auto* sample = ExemplarySamples::createPlainMultiLayerBySLD();
    auto sim1 = test::makeSimulation::TOFRWithRelativeResolution(*sample);
    EXPECT_TRUE(runTest("TOFResolutionComparison_TR", *sim1, 1e-10));
    auto sim2 = test::makeSimulation::TOFRWithPointwiseResolution(*sample);
    EXPECT_TRUE(runTest("TOFResolutionComparison_TP", *sim2, 1e-10));
}

TEST(TESTNAME, BasicSpecular)
{
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createSimpleMagneticLayer());
    for (const std::string polCase : {"PP", "MM"}) {
        auto sim = test::makeSimulation::BasicYPolarizedSpecular(*sample, polCase, false);
        EXPECT_TRUE(runTest("BasicSpecular" + polCase, *sim, 1e-10));
    }
}

TEST(TESTNAME, PolarizedQAngleReflectivity)
{
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createSimpleMagneticLayer());
    for (const std::string polKey : {"PP", "MM"}) {
        auto sim = test::makeSimulation::BasicYPolarizedSpecular(*sample, polKey, true);
        EXPECT_TRUE(runTest("PolarizedQAngleReflectivity" + polKey + "_Q", *sim, 1e-10));
    }
}

TEST(TESTNAME, MagneticRotationReflectivity)
{
    for (const auto roughnessKey : {"Flat", "Tanh", "NevotCroce"}) {
        std::unique_ptr<const MultiLayer> sample(
            ExemplarySamples::createSimpleMagneticRotationWithRoughness(roughnessKey));
        for (const std::string polKey : {"PP", "PM", "MP", "MM"}) {
            auto sim = test::makeSimulation::BasicYPolarizedSpecular(*sample, polKey, false);
            EXPECT_TRUE(
                runTest("MagneticRotationReflectivity" + polKey + "_" + roughnessKey, *sim, 1e-10));
        }
    }
}

TEST(TESTNAME, PolarizedFeNiBilayer)
{
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createFeNiBilayer());
    for (const std::string polKey : {"PP", "MM"}) {
        auto sim1 = test::makeSimulation::BasicYPolarizedSpecular(*sample, polKey, false);
        EXPECT_TRUE(runTest("PolarizedFeNiBilayer" + polKey, *sim1, 1e-7));
        auto sim2 = test::makeSimulation::BasicYPolarizedSpecular(*sample, polKey, true);
        EXPECT_TRUE(runTest("PolarizedFeNiBilayer" + polKey + "_Q", *sim2, 1e-7));
    }
}

TEST(TESTNAME, PolarizedFeNiBilayerTanh)
{
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createFeNiBilayerTanh());
    for (const std::string polKey : {"PP", "MM"}) {
        auto sim1 = test::makeSimulation::BasicYPolarizedSpecular(*sample, polKey, false);
        EXPECT_TRUE(runTest("PolarizedFeNiBilayerTanh" + polKey, *sim1, 1e-7));
        auto sim2 = test::makeSimulation::BasicYPolarizedSpecular(*sample, polKey, true);
        EXPECT_TRUE(runTest("PolarizedFeNiBilayerTanh" + polKey + "_Q", *sim2, 1e-7));
    }
}

TEST(TESTNAME, PolarizedFeNiBilayerNC)
{
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createFeNiBilayerNC());
    for (const std::string polKey : {"PP", "MM"}) {
        auto sim1 = test::makeSimulation::BasicYPolarizedSpecular(*sample, polKey, false);
        EXPECT_TRUE(runTest("PolarizedFeNiBilayerNC" + polKey, *sim1, 1e-7));
        auto sim2 = test::makeSimulation::BasicYPolarizedSpecular(*sample, polKey, true);
        EXPECT_TRUE(runTest("PolarizedFeNiBilayerNC" + polKey + "_Q", *sim2, 1e-7));
    }
}

TEST(TESTNAME, PolarizedFeNiBilayerSpinFlip)
{
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createFeNiBilayerSpinFlip());
    for (const std::string polKey : {"PP", "PM", "MP", "MM"}) {
        auto sim1 = test::makeSimulation::BasicYPolarizedSpecular(*sample, polKey, false);
        EXPECT_TRUE(runTest("PolarizedFeNiBilayerSpinFlip" + polKey, *sim1, 1e-7));
        auto sim2 = test::makeSimulation::BasicYPolarizedSpecular(*sample, polKey, true);
        EXPECT_TRUE(runTest("PolarizedFeNiBilayerSpinFlip" + polKey + "_Q", *sim2, 1e-7));
    }
}

TEST(TESTNAME, PolarizedFeNiBilayerSpinFlipTanh)
{
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createFeNiBilayerSpinFlipTanh());
    for (const std::string polKey : {"PP", "PM", "MP", "MM"}) {
        auto sim1 = test::makeSimulation::BasicYPolarizedSpecular(*sample, polKey, false);
        EXPECT_TRUE(runTest("PolarizedFeNiBilayerSpinFlipTanh" + polKey, *sim1, 1e-7));
        auto sim2 = test::makeSimulation::BasicYPolarizedSpecular(*sample, polKey, true);
        EXPECT_TRUE(runTest("PolarizedFeNiBilayerSpinFlipTanh" + polKey + "_Q", *sim2, 1e-7));
    }
}

TEST(TESTNAME, PolarizedFeNiBilayerSpinFlipNC)
{
    std::unique_ptr<const MultiLayer> sample(ExemplarySamples::createFeNiBilayerSpinFlipNC());
    for (const std::string polKey : {"PP", "PM", "MP", "MM"}) {
        auto sim1 = test::makeSimulation::BasicYPolarizedSpecular(*sample, polKey, false);
        EXPECT_TRUE(runTest("PolarizedFeNiBilayerSpinFlipNC" + polKey, *sim1, 1e-7));
        auto sim2 = test::makeSimulation::BasicYPolarizedSpecular(*sample, polKey, true);
        EXPECT_TRUE(runTest("PolarizedFeNiBilayerSpinFlipNC" + polKey + "_Q", *sim2, 1e-7));
    }
}

TEST(TESTNAME, Depthprobe)
{
    auto* sample = ExemplarySamples::createHomogeneousMultilayer();
    auto sim = test::makeSimulation::BasicDepthprobe(*sample);
    EXPECT_TRUE(runTest("DepthprobeTest", *sim, 1e-10));
}

#endif // GUI_STD_TEST
#endif // PYTHON_STD_TEST

#endif // BORNAGAIN_TESTS_SUITE_COMMON_TESTSUITE_H
