# This file implements the process of making source distribution tarballs. It expects to find a file list in
# the cmake cache variable dist_files.
#
# Given all of these files, the procedure is to:
# 1. Remove any existing dist directory and make a new one.
# 2. Copy of all the files in dist_manifest.text and ${dist_generated}
#    into the dist directory.
# 3. Create the tarball and compress it with gzip and bzip2.
# 4. Then remove the dist directory.

include(${CMAKE_MODULE_PATH}/DistCommon.cmake)

function(find_automake AUTOMAKE_VAR ACLOCAL_VAR AUTOMAKE_VERSION_VAR)
    find_program(AUTOMAKE automake)
    execute_process(
            COMMAND ${AUTOMAKE} --version
            RESULT_VARIABLE AUTOMAKE_RESULT
            OUTPUT_VARIABLE AUTOMAKE_OUTPUT
            ERROR_VARIABLE AUTOMAKE_ERROR
    )

    set(NEED_OVERRIDE FALSE)
    if(AUTOMAKE)
        string(REGEX REPLACE ".*automake \\(GNU automake\\) ([0-9]\\.[0-9]+).*" "\\1" AUTOMAKE_VERSION "${AUTOMAKE_OUTPUT}")
        find_program(ACLOCAL aclocal)
        if(NOT ACLOCAL)
            message(FATAL_ERROR "Found ok version of automake, but can't find aclocal")
        endif()
    else()
        message(FATAL_ERROR "Can't find 'automake' or 'automake-1.11'")
        message("  You can set AUTOTOOLS_IN_DIST=OFF to exclude autotools support.")
    endif()
    set(${AUTOMAKE_VAR} ${AUTOMAKE} PARENT_SCOPE)
    set(${ACLOCAL_VAR} ${ACLOCAL} PARENT_SCOPE)
    set(${AUTOMAKE_VERSION_VAR} ${AUTOMAKE_VERSION} PARENT_SCOPE)
endfunction()

function(make_dist PACKAGE_PREFIX GNUCASH_SOURCE_DIR BUILD_SOURCE_DIR)

    # -- Remove any existing packaging directory.
    file(REMOVE_RECURSE ${PACKAGE_PREFIX})

    if (EXISTS ${PACKAGE_PREFIX})
        message(FATAL_ERROR "Unable to remove existing dist directory \"${PACKAGE_PREFIX}\". Cannot continue.")
    endif()

    # -- Copy in distributed files
    foreach(file ${dist_files})
        if(NOT EXISTS ${GNUCASH_SOURCE_DIR}/${file})
            message(FATAL_ERROR "Can't find dist file ${GNUCASH_SOURCE_DIR}/${file}")
        endif()
        get_filename_component(dir ${file} DIRECTORY)
        file(MAKE_DIRECTORY ${PACKAGE_PREFIX}/${dir})
        file(COPY ${GNUCASH_SOURCE_DIR}/${file} DESTINATION ${PACKAGE_PREFIX}/${dir})
    endforeach()

    cmake_policy(SET CMP0012 NEW)

    # When making a dist tarball including autotools files
    # the Makefile.in files may have to be generated by running autogen.sh
    # The toplevel cmake configuration determines whether this is required or not
    if(AUTOGEN)
        find_automake(AUTOMAKE ACLOCAL AUTOMAKE_VERSION)

        # -- Run autogen.sh to cause Makefile.in files to be created.
        file(COPY ${GNUCASH_SOURCE_DIR}/autogen.sh DESTINATION ${PACKAGE_PREFIX})
        execute_process(
                COMMAND ${SHELL} -c ./autogen.sh
                WORKING_DIRECTORY ${PACKAGE_PREFIX}
                RESULT_VARIABLE AUTOGEN_RESULT
                OUTPUT_VARIABLE AUTOGEN_OUTPUT
        )
        if(NOT ${AUTOGEN_RESULT} STREQUAL "0")
            message(FATAL_ERROR "autogen.sh step failed: ${AUTOGEN_RESULT}")
        endif()

        # -- Remove autogen files as they are not distributed.
        file(REMOVE ${PACKAGE_PREFIX}/autogen.sh)
        file(REMOVE_RECURSE ${PACKAGE_PREFIX}/autom4te.cache)

        # -- Autogen.sh creates some files as symbolic links that we turn into real files here.
        if(NOT WIN32) # No symbolic links on Windows
            set(LINKS config.guess config.sub COPYING INSTALL install-sh ltmain.sh missing)
            foreach(link ${LINKS})
                get_filename_component(realpath ${PACKAGE_PREFIX}/${link} REALPATH)
                file(REMOVE ${PACKAGE_PREFIX}/${link})
                file(COPY ${realpath} DESTINATION ${PACKAGE_PREFIX})
            endforeach(link)
        endif()

    endif()


    # -- Create the tarballs.

    execute_process_and_check_result(
            COMMAND ${CMAKE_COMMAND} -E tar zcf ${PACKAGE_PREFIX}.tar.gz ${PACKAGE_PREFIX}
            WORKING_DIRECTORY .
            ERROR_MSG "tar command to create ${PACKAGE_PREFIX}.tar.gz failed."
    )

    execute_process_and_check_result(
            COMMAND ${CMAKE_COMMAND} -E tar jcf ${PACKAGE_PREFIX}.tar.bz2 ${PACKAGE_PREFIX}
            WORKING_DIRECTORY .
            ERROR_MSG "tar command to create ${PACKAGE_PREFIX}.tar.gz failed."
    )

    # -- Clean up packaging directory.

    file(REMOVE_RECURSE ${PACKAGE_PREFIX})

    if(EXISTS ${PACKAGE_PREFIX})
        message(WARNING "Could not remove packaging directory '${PACKAGE_PREFIX}'")
    endif()

    # -- All done.

    message("\n\nDistributions ${PACKAGE_PREFIX}.tar.gz and ${PACKAGE_PREFIX}.tar.bz2 created.\n\n")
endfunction()

 make_dist(${PACKAGE_PREFIX} ${GNUCASH_SOURCE_DIR} ${BUILD_SOURCE_DIR})
