#
# Copyright (C) 2018 Binero AB.
#
# Author: Tobias Urdin <tobias.urdin@binero.se>
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.
#
# == Class: neutron::agents::bgp_dragent
#
# Install and configure neutron BGP dragent from Neutron Dynamic Routing.
#
# === Parameters:
#
# [*package_ensure*]
#   (Optional) The state of the package.
#   Defaults to 'present'
#
# [*enabled*]
#   (Optional) The state of the service.
#   Defaults to true
#
# [*manage_service*]
#   (Optional) Whether to start/stop the service.
#   Defaults to true
#
# [*bgp_speaker_driver*]
#   (Optional) The BGP speaker driver to use.
#   Defaults to 'neutron_dynamic_routing.services.bgp.agent.driver.os_ken.driver.OsKenBgpDriver'
#
# [*bgp_router_id*]
#   (Optional) The BGP router ID.
#   Defaults to $facts['networking']['ip']
#
# [*purge_config*]
#   (Optional) Whether to set only the specified config options in the BGP dragent config.
#   Defaults to false
#
class neutron::agents::bgp_dragent (
  Stdlib::Ensure::Package $package_ensure = 'present',
  Boolean $enabled                        = true,
  Boolean $manage_service                 = true,
  $bgp_speaker_driver                     = 'neutron_dynamic_routing.services.bgp.agent.driver.os_ken.driver.OsKenBgpDriver',
  $bgp_router_id                          = $facts['networking']['ip'],
  Boolean $purge_config                   = false,
) {
  include neutron::deps
  include neutron::params

  resources { 'neutron_bgp_dragent_config':
    purge => $purge_config,
  }

  neutron_bgp_dragent_config {
    'bgp/bgp_speaker_driver': value => $bgp_speaker_driver;
    'bgp/bgp_router_id':      value => $bgp_router_id;
  }

  package { 'neutron-bgp-dragent':
    ensure => $package_ensure,
    name   => $neutron::params::bgp_dragent_package,
    tag    => ['openstack', 'neutron-package'],
  }

  if $manage_service {
    if $enabled {
      $service_ensure = 'running'
    } else {
      $service_ensure = 'stopped'
    }
    service { 'neutron-bgp-dragent':
      ensure => $service_ensure,
      name   => $neutron::params::bgp_dragent_service,
      enable => $enabled,
      tag    => 'neutron-service',
    }
    Neutron_bgp_dragent_config<||> ~> Service['neutron-bgp-dragent']
  }
}
